/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "Log.h"
#include "Subdivision.h"
#include "TouchAreaSubdivision.h"
#include "WindowSystems/TouchAreaInputEventDispatcher.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"

using namespace InputEventProcessing;
using namespace LayerManagerCalibration;

TouchAreaInputEventDispatcher::TouchAreaInputEventDispatcher()
    : InputEventDispatcher()
{
}

TouchAreaInputEventDispatcher::~TouchAreaInputEventDispatcher()
{
}

void TouchAreaInputEventDispatcher::processTouchEvent(Subdivision* subdivision,
                                                      struct evdev_input_device* device,
                                                      coordinate& calibratedCoordinate,
                                                      struct TouchEvent& touchEvent)
{
    LOG_DEBUG("TouchAreaInputEventDispatcher",
              "Process touch event for subdivision, name="
              << subdivision->getName()
              << ", coordinate x="
              << calibratedCoordinate.x << ", y=" << calibratedCoordinate.y);

    // generate a coordinate relative to the origin of the subdivision
    coordinate relativeCoordinate = {
        calibratedCoordinate.x - subdivision->getTopLeftCoordinate().x,
        calibratedCoordinate.y - subdivision->getTopLeftCoordinate().y,
    };

    // obtain the screen resolution
    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (inputEvent == NULL)
    {
        LOG_WARNING("TouchAreaInputEventDispatcher",
                    "Failed to process touch event");
    }
    else
    {
        int screenWidth = 0;
        int screenHeight = 0;
        inputEvent->getScreenResolution(device->displayID, screenWidth, screenHeight);

        // scale the relative coordinate to the resolution of the display
        coordinate logicalCoordinate = {
            (uint) (relativeCoordinate.x / float(subdivision->getWidth()) * screenWidth),
            (uint) (relativeCoordinate.y / float(subdivision->getHeight()) * screenHeight),
        };

        LOG_DEBUG("TouchAreaInputEventDispatcher",
                  "Logical coordinate, "
                  "x=" << logicalCoordinate.x << ", "
                  "y=" << logicalCoordinate.y);

        if (getListener() != NULL)
        {
            TouchAreaSubdivision *touchSubdivision = static_cast<TouchAreaSubdivision*>(subdivision);

            coordinate offsetCoordinate = {
                logicalCoordinate.x,
                logicalCoordinate.y
            };

            if (touchSubdivision != NULL)
            {
                offsetCoordinate.x = logicalCoordinate.x + touchSubdivision->getDisplayOffset().x;
                offsetCoordinate.y = logicalCoordinate.y + touchSubdivision->getDisplayOffset().y;
            }

            if (touchEvent.state == INPUT_STATE_PRESSED)
            {
                if (!getState().getSubdivisionSlotPressed(device,
                                                          subdivision,
                                                          touchEvent.slot))
                {
                    getState().setSubdivisionSlotPressed(device,
                                                         subdivision,
                                                         touchEvent.slot,
                                                         true);

                    if (touchEvent.isSingleTouch)
                    {
                        getListener()->generateButtonEvent(device,
                                                           touchEvent.time,
                                                           offsetCoordinate.x,
                                                           offsetCoordinate.y,
                                                           touchEvent.button,
                                                           WL_POINTER_BUTTON_STATE_PRESSED);
                    }
                    else
                    {
                        getListener()->generateTouchEvent(device,
                                                          EVDEV_ABSOLUTE_MT_DOWN,
                                                          offsetCoordinate);
                    }
                }
            }

            if (touchEvent.state == INPUT_STATE_RELEASED)
            {
                if (getState().getSubdivisionSlotPressed(device,
                                                         subdivision,
                                                         touchEvent.slot))
                {
                    getState().setSubdivisionSlotPressed(device,
                                                         subdivision,
                                                         touchEvent.slot,
                                                         false);

                    if (touchEvent.isSingleTouch)
                    {
                        getListener()->generateButtonEvent(device,
                                                           touchEvent.time,
                                                           offsetCoordinate.x,
                                                           offsetCoordinate.y,
                                                           touchEvent.button,
                                                           WL_POINTER_BUTTON_STATE_RELEASED);
                    }
                    else
                    {
                        getListener()->generateTouchEvent(device,
                                                          EVDEV_ABSOLUTE_MT_UP,
                                                          offsetCoordinate);
                    }
                }
            }

            if (touchEvent.state == INPUT_STATE_MOTION)
            {
                if (touchEvent.isSingleTouch)
                {
                    getListener()->generateMotionEvent(device, EVDEV_ABSOLUTE_MOTION, offsetCoordinate);
                }
                else
                {
                    getListener()->generateMotionEvent(device, EVDEV_ABSOLUTE_MT_MOTION, offsetCoordinate);
                }
            }
        }
    }
}

void TouchAreaInputEventDispatcher::processEnter(LayerManagerCalibration::Subdivision* subdivision,
                                                 struct evdev_input_device* device,
                                                 LayerManagerCalibration::coordinate& calibratedCoordinate,
                                                 struct TouchEvent& touchEvent)
{
    LOG_DEBUG("TouchAreaInputEventDispatcher",
              "Process enter for subdivision, "
              "name=" << subdivision->getName());

    if (!getState().getSubdivisionSlotPressed(device, subdivision, touchEvent.slot))
    {
        getState().setSubdivisionSlotPressed(device,
                                             subdivision,
                                             touchEvent.slot,
                                             true);

        // obtain the screen resolution
        WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
        if (inputEvent == NULL)
        {
            LOG_WARNING("TouchAreaInputEventDispatcher",
                        "Failed to process touch event");
        }
        else
        {
            if (getListener() != NULL)
            {
                TouchAreaSubdivision *touchSubdivision = static_cast<TouchAreaSubdivision*>(subdivision);
                int screenWidth = 0;
                int screenHeight = 0;
                inputEvent->getScreenResolution(device->displayID, screenWidth, screenHeight);

                // generate a coordinate relative to the origin of the subdivision
                coordinate relativeCoordinate = {
                    calibratedCoordinate.x - subdivision->getTopLeftCoordinate().x,
                    calibratedCoordinate.y - subdivision->getTopLeftCoordinate().y,
                };

                // scale the relative coordinate to the resolution of the display
                coordinate logicalCoordinate = {
                    (uint) (relativeCoordinate.x / float(subdivision->getWidth()) * screenWidth),
                    (uint) (relativeCoordinate.y / float(subdivision->getHeight()) * screenHeight),
                };

                if (touchSubdivision != NULL)
                {
                    logicalCoordinate.x = logicalCoordinate.x + touchSubdivision->getDisplayOffset().x;
                    logicalCoordinate.y = logicalCoordinate.y + touchSubdivision->getDisplayOffset().y;
                }

                LOG_DEBUG("TouchAreaInputEventDispatcher",
                         "Logical coordinate, "
                         "x=" << logicalCoordinate.x << ", "
                         "y=" << logicalCoordinate.y);

                if (touchEvent.isSingleTouch)
                {
                    getListener()->generateButtonEvent(device,
                                                       touchEvent.time,
                                                       logicalCoordinate.x,
                                                       logicalCoordinate.y,
                                                       touchEvent.button,
                                                       WL_POINTER_BUTTON_STATE_PRESSED);
                }
                else
                {
                    getListener()->generateTouchEvent(device,
                                                      EVDEV_ABSOLUTE_MT_DOWN,
                                                      logicalCoordinate);
                }
            }
        }
    }
}

void TouchAreaInputEventDispatcher::processLeave(LayerManagerCalibration::Subdivision* subdivision,
                                                 struct evdev_input_device* device,
                                                 LayerManagerCalibration::coordinate& calibratedCoordinate,
                                                 struct TouchEvent& touchEvent)
{
    LOG_DEBUG("TouchAreaInputEventDispatche",
              "subdivision, name=" << subdivision->getName());

    if (getState().getSubdivisionSlotPressed(device,
                                             subdivision,
                                             touchEvent.slot))
    {
        getState().setSubdivisionSlotPressed(device,
                                             subdivision,
                                             touchEvent.slot,
                                             false);

        // obtain the screen resolution
        WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
        if (inputEvent == NULL)
        {
            LOG_WARNING("TouchAreaInputEventDispatcher",
                        "Failed to process touch event");
        }
        else
        {
            if (getListener() != NULL)
            {
                coordinate rawClippedCoordinate;
                subdivision->clipCoordinate(calibratedCoordinate, rawClippedCoordinate);

                TouchAreaSubdivision *touchSubdivision = static_cast<TouchAreaSubdivision*>(subdivision);
                int screenWidth = 0;
                int screenHeight = 0;
                inputEvent->getScreenResolution(device->displayID, screenWidth, screenHeight);

                // generate a coordinate relative to the origin of the subdivision
                coordinate relativeCoordinate = {
                    rawClippedCoordinate.x - subdivision->getTopLeftCoordinate().x,
                    rawClippedCoordinate.y - subdivision->getTopLeftCoordinate().y,
                };

                // scale the relative coordinate to the resolution of the display
                coordinate logicalCoordinate = {
                    (uint) (relativeCoordinate.x / float(subdivision->getWidth()) * screenWidth),
                    (uint) (relativeCoordinate.y / float(subdivision->getHeight()) * screenHeight),
                };

                if (touchSubdivision != NULL)
                {
                    logicalCoordinate.x = logicalCoordinate.x + touchSubdivision->getDisplayOffset().x;
                    logicalCoordinate.y = logicalCoordinate.y + touchSubdivision->getDisplayOffset().y;
                }

                LOG_DEBUG("TouchAreaInputEventDispatcher",
                          "Logical coordinate, "
                          "x=" << logicalCoordinate.x << ", "
                          "y=" << logicalCoordinate.y);

                if (touchEvent.isSingleTouch)
                {
                    getListener()->generateButtonEvent(device,
                                                       touchEvent.time,
                                                       logicalCoordinate.x,
                                                       logicalCoordinate.y,
                                                       touchEvent.button,
                                                       WL_POINTER_BUTTON_STATE_RELEASED);
                }
                else
                {
                    getListener()->generateTouchEvent(device,
                                                      EVDEV_ABSOLUTE_MT_UP,
                                                      logicalCoordinate);
                }
            }
        }
    }
}
